<?php

namespace xrPublisher\Core;


class Plugin
{
	public function init() {
		// Add actions and filters
		add_filter( 'run_wptexturize', '__return_false' );
		add_filter('upload_mimes', array( $this, 'xr_publisher_add_file_types_to_uploads'), 10, 4);
		add_filter( 'wp_check_filetype_and_ext',  array( $this, 'xr_publisher_check_for_usdz'), 10, 4 );
		add_action('wp_enqueue_scripts',  array( $this, 'xr_publisher_frontend_assets'));
		add_action( 'rest_api_init',  array( $this, 'callAlchemy' ));
		add_action('enqueue_block_assets',  array( $this, 'xr_publisher_editor_assets'));
		//Enqueue JavaScript and CSS for xr_publisher_loader_init
		add_action( 'wp_enqueue_scripts',  array( $this, 'xr_publisher_enqueue_xr_publisher_loader_init'), 10 );
		add_filter( 'the_content', array( $this, 'remove_block_from_archive' ), 10 );
		add_filter( 'wp_kses_allowed_html', array( $this, 'allow_xr_publisher_web_components' ), 10, 2 );
		add_action( 'enqueue_block_editor_assets', array( $this, 'load_xr_publisher_translations_blocks' ) );
		add_action( 'init', array( $this, 'load_xr_publisher_textdomain' ) );

		// Add new actions for user profile fields and saving
		add_action('show_user_profile', array($this, 'add_custom_user_profile_fields'));
		add_action('edit_user_profile', array($this, 'add_custom_user_profile_fields'));
		add_action('personal_options_update', array($this, 'save_custom_user_profile_fields'));
		add_action('edit_user_profile_update', array($this, 'save_custom_user_profile_fields'));
		add_action('rest_api_init', array( $this, 'register_room_count_api_routes' ));
		add_action('rest_api_init', array( $this, 'register_turn_credential_endpoint' ));

		// add_action( 'init', array( $this, 'register_room_count_post_meta' ) );
		// Block Category
		add_filter( 'block_categories_all', array( $this, 'add_spatial_block_category'), 10, 2 );
		add_action('admin_enqueue_scripts', array( $this, 'xr_publisher_assets_tab_assets' ) );
		add_action('admin_footer', array($this,'custom_media_tab_template'));
		add_action('rest_api_init', array($this, 'register_toybox_assets_endpoint'));
		add_action('rest_api_init', array($this, 'register_toybox_categories_endpoint'));

	}

	// Callback function for the REST API endpoint
	function get_toybox_assets($request) {
		if (!$this->check_user_permissions($request)) {
			return new \WP_Error('toybox_api_error', 'Sorry, you are not allowed to do that.', array('status' => 403));
		}

		$api_key = get_option('xr_publisher_toyboxApiKey');
		$limit = $request->get_param('limit');
		$offset = $request->get_param('offset');
		$search = $request->get_param('search');
		$categories = $request->get_param('categories');

		$url = "https://cfdb.sxpdigital.workers.dev/assets-by-key?limit={$limit}&offset={$offset}&search={$search}&categories={$categories}";

		$args = array(
			'headers'     => array(
				'Content-Type' => 'application/json',
				'Authorization' => 'Bearer ' . $api_key,
			),
		); 
		
		$response = wp_remote_request($url, $args);

		if (is_wp_error($response)) {
			return new \WP_Error('toybox_api_error', 'Error retrieving assets from Toybox API', array('status' => 500));
		}

		$body = wp_remote_retrieve_body($response);
		$data = json_decode($body, true);

		// // make a new nonce to send back to the client
		// $data['nonce'] = wp_create_nonce('wp_rest');

		return new \WP_REST_Response($data, 200);
	}

	function get_toybox_categories($request) {
		if (!$this->check_user_permissions($request)) {
			return new \WP_Error('toybox_api_error', 'Sorry, you are not allowed to do that.', array('status' => 403));
		}
	
		$api_key = get_option('xr_publisher_toyboxApiKey');
	
		$url = "https://cfdb.sxpdigital.workers.dev/categories-by-key";
	
		$args = array(
			'headers' => array(
				'Content-Type' => 'application/json',
				'Authorization' => 'Bearer ' . $api_key,
			),
		);
	
		$response = wp_remote_request($url, $args);
	
		if (is_wp_error($response)) {
			return new \WP_Error('toybox_api_error', 'Error retrieving categories from Toybox API', array('status' => 500));
		}
	
		$body = wp_remote_retrieve_body($response);
		$data = json_decode($body, true);
	
		return new \WP_REST_Response($data, 200);
	}
	
	// Check user permissions using nonce verification
	function check_user_permissions($request) {
		$nonce = $request->get_header('X-WP-Nonce');
		if (!wp_verify_nonce($nonce, 'wp_rest')) {
			return false;
		}
		return current_user_can('edit_posts');
	}

	function register_toybox_assets_endpoint() {
		register_rest_route('toybox/v1', '/assets', array(
			'methods' => 'GET',
			'callback' => array($this, 'get_toybox_assets'),
			'permission_callback' => array($this, 'check_user_permissions'),
		));
	}

	function register_toybox_categories_endpoint() {
		register_rest_route('toybox/v1', '/categories', array(
			'methods' => 'GET',
			'callback' => array($this, 'get_toybox_categories'),
			'permission_callback' => array($this, 'check_user_permissions'),
		));
	}
	
	function custom_media_tab_template() {
		?>
		<script type="text/html" id="tmpl-custom-media-tab">
			<div class="custom-media-tab-content">
				<div class="search-container">
					<form class="search-form">
						<input type="text" class="search-input" placeholder="Search assets...">
						<button type="submit" class="button button-primary search-submit">Search</button>
						<button type="button" class="button button-secondary clear-search">Clear</button>
					</form>
				</div>
				<div class="category-filters">
					<!-- Category filter buttons will be rendered here -->
				</div>
				<ul class="asset-list" style="display: grid; grid-template-columns: repeat(auto-fill, 150px); gap: 10px;"></ul>
				<button class="button button-primary assets-load-more">Load More</button>
			</div>
		</script>
	
		<script type="text/html" id="tmpl-asset-template">
			<div class="asset-details" style="width: 150px; height: 150px; display: flex; align-items: center; justify-content: center; cursor: pointer;">
				<img src="{{ model.attributes.thumburl }}" alt="{{ model.attributes.filename }}" style="max-width: 100%; max-height: 100%; object-fit: contain;" />
			</div>
		</script>
		<?php
	}
	
	function xr_publisher_assets_tab_assets($hook) {
		// Only enqueue the script on post edit screens
		if ('post.php' !== $hook && 'post-new.php' !== $hook) {
			return;
		}
		wp_enqueue_script('wp-element');
		wp_enqueue_script('xr-publisher-assets-tab', plugin_dir_url(__FILE__) . '../inc/utils/media-modal/media-modal.js', array('jquery', 'media-views', 'wp-element'), '1.0', true);
		wp_localize_script('xr-publisher-assets-tab', 'xrPublisherAssetsTab', array(
			'toybox_nonce' => wp_create_nonce('wp_rest'),
		));		
		// Enqueue the CSS file
		wp_enqueue_style('xr-publisher-assets-tab-css', plugin_dir_url(__FILE__) . '../inc/utils/media-modal/media-modal.css', array(), '1.0');

	}
	
	function register_turn_credential_endpoint() {
		register_rest_route('xr-publisher/v1', '/turn-credentials/', array(
			'methods' => \WP_REST_Server::READABLE,
			'callback' => array( $this, 'get_turn_credentials'),
			'permission_callback' => array( $this, 'is_user_allowed_turn')
		));
	}

	function is_user_allowed_turn() {
		// check xr_publisher_mp_multiplayerAccess option if we even need to check this
		if (get_option('xr_publisher_mp_multiplayerAccess') === 'loggedIn') {
			return is_user_logged_in();
		} else {
			return true;
		}
	}

	function get_turn_credentials($request) {
		$nonce = $request->get_header('X-WP-Nonce');
		if (!wp_verify_nonce($nonce, 'wp_rest')) {
			return new \WP_Error('invalid_nonce', 'Invalid nonce', array('status' => 403));
		}
	
		// Get the API key from 'xr_publisher_mp_turnServerKey'
		$apiKey = get_option('xr_publisher_mp_turnServerKey');
	
		// URL of your Cloudflare Worker
		$workerUrl = 'https://cfdb.sxpdigital.workers.dev/generate-turn-credentials';
	
		// Make a request to the Cloudflare Worker to get TURN credentials
		$response = wp_remote_post($workerUrl, array(
			'headers' => array(
				'Content-Type' => 'application/json',
				'Authorization' => 'Bearer ' . $apiKey,
			),
		));
	
		if (is_wp_error($response)) {
			$error_message = $response->get_error_message();
			return new \WP_Error('request_failed', $error_message, array('status' => 400));
		}
	
		$body = wp_remote_retrieve_body($response);
		$credentials = json_decode($body, true);
	
		return new \WP_REST_Response($credentials, 200);
	}
	
	
	// _updateRoomCountInDatabase(roomCount) {
	// 	const postId = userData.currentPostId // Implement this method based on how you determine the post ID
	// 	const apiUrl = `/wp-json/xr-publisher/v1/update-room-count/${postId}`;
	// 	const data = { count: roomCount };

	// 	fetch(apiUrl, {
	// 		method: 'POST',
	// 		headers: {
	// 			'Content-Type': 'application/json',
	// 			'X-WP-Nonce': userData.nonce,
	// 		},
	// 		body: JSON.stringify(data)
	// 	})
	// 	.then(response => response.json())
	// 	.then(data => console.log('Room count updated:', data))
	// 	.catch((error) => {
	// 		console.error('Error updating room count:', error);
	// 	});
	// }

	function register_room_count_api_routes() {
		register_rest_route('xr-publisher/v1', '/add-room-count/(?P<id>\d+)', array(
			'methods' => 'POST',
			'callback' => array( $this, 'update_room_count'),
			'permission_callback' => array( $this, 'check_add_token'),
			'args' => array(
				'action' => array(
					'required' => true,
					'type' => 'string',
					'enum' => array('add', 'subtract'),
				),
			),
		));
		register_rest_route('xr-publisher/v1', '/subtract-room-count/(?P<id>\d+)', array(
			'methods' => 'POST',
			'callback' => array( $this, 'update_room_count'),
			'permission_callback' => array( $this, 'check_subtract_token'),
			'args' => array(
				'action' => array(
					'required' => true,
					'type' => 'string',
					'enum' => array('add', 'subtract'),
				),
			),
		));
		register_rest_route('xr-publisher/v1', '/get-room-count/(?P<id>\d+)', array(
			'methods' => 'GET',
			'callback' => array( $this, 'get_room_count'),
			'permission_callback' => '__return_true', // Adjust the permission callback as needed
		));
		register_rest_route('xr-publisher/v1', '/send-heartbeat/(?P<id>\d+)', array(
			'methods' => 'POST',
			'callback' => array( $this, 'handle_heartbeat'),
			'permission_callback' => '__return_true'
		));
		register_rest_route('xr-publisher/v1', '/handle-heart-check/(?P<id>\d+)', array(
			'methods' => 'POST',
			'callback' => array( $this, 'handle_heart_check'),
			'permission_callback' => '__return_true'
		));	
	}
	function handle_heart_check($request) {
		$post_id = $request['id'];
		$participant_name = $request['displayName'];
		$site_url = get_site_url();
		$user_id = get_current_user_id();
	
		// Construct the key for the Cloudflare KV store
		$kv_key = "heartbeat|{$site_url}|{$post_id}|{$participant_name}";
	
		// Current timestamp as heartbeat
		$heartbeat_timestamp = time();
	
		// Replace with the URL of your Cloudflare Worker
		$cloudflare_worker_url = "https://room-balancer.sxpdigital.workers.dev/handle-heart-check/";
	
		$response = wp_remote_post($cloudflare_worker_url, array(
			'headers' => array('Content-Type' => 'application/json'),
			'body' => wp_json_encode(array(
				'key' => $kv_key,
				'timestamp' => $heartbeat_timestamp,
			)),
			'timeout' => 45,
		));
	
		if (is_wp_error($response)) {
			return new \WP_REST_Response('Error sending heartbeat', 500);
		}
	
		// return the data of the response
		return new \WP_REST_Response($response, 200);
	
	}

	function handle_heartbeat($request) {
		$post_id = $request['id'];
		$participant_name = $request['displayName'];
		$site_url = get_site_url();
		$user_id = get_current_user_id();
	
		// Construct the key for the Cloudflare KV store
		$kv_key = "heartbeat|{$site_url}|{$post_id}|{$participant_name}";
	
		// Current timestamp as heartbeat
		$heartbeat_timestamp = time();
	
		// Replace with the URL of your Cloudflare Worker
		$cloudflare_worker_url = "https://room-balancer.sxpdigital.workers.dev/heartbeat/";
	
		$response = wp_remote_post($cloudflare_worker_url, array(
			'headers' => array('Content-Type' => 'application/json'),
			'body' => wp_json_encode(array(
				'key' => $kv_key,
				'timestamp' => $heartbeat_timestamp,
			)),
			'timeout' => 45,
		));
	
		if (is_wp_error($response)) {
			return new \WP_REST_Response('Error sending heartbeat', 500);
		}
	
		return new \WP_REST_Response('Heartbeat updated', 200);
	}

	function add_spatial_block_category( $categories, $post ) {
		return array_merge(
			$categories,
			array(
				array(
					'slug'  => 'spatial',
					'title' => __( 'Spatial', 'xr-publisher' ),
					'icon'  => 'block-default',
				),
			)
		);
	}
	
	function get_room_count($request) {
		return true;
	}
	
	// Callback function to make a request to the Cloudflare Worker
	function update_room_count($request) {
		$post_id = $request['id'];
		$action = $request['action'];
		$clientId = $request['clientId'] ? $request['clientId'] : null;
		$cached_room_count = get_transient("room_count_{$post_id}");
		// clear the transient if it exists because we are updating the room count
		if ($cached_room_count) {
			delete_transient("room_count_{$post_id}");
		}

		// Replace with the URL of your Cloudflare Worker
		$cloudflare_worker_url = "https://room-balancer.sxpdigital.workers.dev";
	
		$response = wp_remote_post($cloudflare_worker_url, array(
			'body' => wp_json_encode(array(
				'siteUrl' => get_site_url(),
				'postId' => $post_id,
				'action' => $action,
				'userId' => $clientId
			)),
			'headers' => array('Content-Type' => 'application/json'),
			'timeout'     => 45,
		));
	
		if (is_wp_error($response)) {
			return new \WP_REST_Response('Error communicating with the room load balancer', 500);
		}
	
		$body = wp_remote_retrieve_body($response);
		return new \WP_REST_Response($body, 200);
	}

	public function allow_xr_publisher_web_components($tags, $context) {
		// check others! @todo
		if ($context === 'post') {
			$tags['three-object-block'] = array(
				'class' => true,
				'align' => true,
				'scale' => true,
				'positionx' => true,
				'positiony' => true,
				'positionz' => true,
				'rotationy' => true,
				'threeobjecturl' => true,
				'threepreviewimage' => true,
				'hdr' => true,
				'devicetarget' => true,
				'animations' => true
			);
			$tags['three-environment-block'] = array(
				'class' => true,
				'align' => true,
				'scale' => true,
				'positionx' => true,
				'positiony' => true,
				'positionz' => true,
				'rotationy' => true,
				'threeobjecturl' => true,
				'threepreviewimage' => true,
				'hdr' => true,
				'devicetarget' => true,
				'animations' => true
			);
			$tags['three-spawn-point-block'] = array(
				'class' => true,
				'positionx' => true,
				'positiony' => true,
				'positionz' => true,
				'rotationx' => true,
				'rotationy' => true,
				'rotationz' => true 
			);
	
			$tags['three-model-block'] = array(
				'class' => true,
				'threeobjecturl' => true,
				'scalex' => true,
				'scaley' => true,
				'scalez' => true,
				'positionx' => true,
				'positiony' => true,
				'positionz' => true,
				'rotationx' => true,
				'rotationy' => true,
				'rotationz' => true,
				'animations' => true,
				'collidable' => true,
				'alt' => true
			);
	
			$tags['three-npc-block'] = array(
				'class' => true,
				'threeobjecturl' => true,
				'positionx' => true,
				'positiony' => true,
				'positionz' => true,
				'rotationx' => true,
				'rotationy' => true,
				'rotationz' => true,
				'name' => true,
				'defaultmessage' => true,
				'personality' => true,
				'objectawareness' => true
			);
	
			$tags['three-sky-block'] = array(
				'class' => true,
				'skyUrl' => true,
				'distance' => true,
				'rayleigh' => true,
				'sunpositionx' => true,
				'sunpositiony' => true,
				'sunpositionz' => true
			);
	
			$tags['three-audio-block'] = array(
				'class' => true,
				'audiourl' => true,
				'scalex' => true,
				'scaley' => true,
				'scalez' => true,
				'positionx' => true,
				'positiony' => true,
				'positionz' => true,
				'rotationx' => true,
				'rotationy' => true,
				'rotationz' => true,
				'autoplay' => true,
				'loop' => true,
				'volume' => true,
				'positional' => true,
				'coneinnerangle' => true,
				'coneouterangle' => true,
				'coneoutergain' => true,
				'distancemodel' => true,
				'maxdistance' => true,
				'refdistance' => true,
				'rollofffactor' => true
			);
	
			$tags['three-image-block'] = array(
				'class' => true,
				'imageurl' => true,
				'scalex' => true,
				'scaley' => true,
				'scalez' => true,
				'positionx' => true,
				'positiony' => true,
				'positionz' => true,
				'rotationx' => true,
				'rotationy' => true,
				'rotationz' => true,
				'aspectheight' => true,
				'aspectwidth' => true,
				'transparent' => true
			);
	
			$tags['three-light-block'] = array(
				'class' => true,
				'positionx' => true,
				'positiony' => true,
				'positionz' => true,
				'rotationx' => true,
				'rotationy' => true,
				'rotationz' => true,
				'type' => true,
				'color' => true,
				'intensity' => true,
				'distance' => true,
				'decay' => true,
				'targetx' => true,
				'targety' => true,
				'targetz' => true,
				'angle' => true,
				'penumbra' => true
			);
	
			$tags['three-networking-block'] = array(
				'class' => true,
				'participantlimit' => true
			);
	
			$tags['three-portal-block'] = array(
				'class' => true,
				'threeobjecturl' => true,
				'destinationurl' => true,
				'scalex' => true,
				'scaley' => true,
				'scalez' => true,
				'positionx' => true,
				'positiony' => true,
				'positionz' => true,
				'rotationx' => true,
				'rotationy' => true,
				'rotationz' => true,
				'animations' => true,
				'label' => true,
				'labeloffsetx' => true,
				'labeloffsety' => true,
				'labeloffsetz' => true,
				'labeltextcolor' => true
			);
	
			$tags['three-text-block'] = array(
				'class' => true,
				'textcontent' => true,
				'positionx' => true,
				'positiony' => true,
				'positionz' => true,
				'rotationx' => true,
				'rotationy' => true,
				'rotationz' => true,
				'scalex' => true,
				'scaley' => true,
				'scalez' => true,
				'textcolor' => true
			);
	
			$tags['three-video-block'] = array(
				'class' => true,
				'videourl' => true,
				'scalex' => true,
				'scaley' => true,
				'scalez' => true,
				'positionx' => true,
				'positiony' => true,
				'positionz' => true,
				'rotationx' => true,
				'rotationy' => true,
				'rotationz' => true,
				'autoplay' => true,
				'custommodel' => true,
				'aspectheight' => true,
				'aspectwidth' => true,
				'videocontrolsenabled' => true,
				'modelurl' => true
			);
	
		}
		return $tags;
	}
	

	/**
	 * Adds custom spatial user profile fields
	 */
	public function add_custom_user_profile_fields($user) {
		?>
		<h3><?php esc_html_e("Spatial Profile Information"); ?></h3>
		<table class="form-table">
			<tr>
				<th>
					<label for="user_data_vrm"><?php esc_html_e("VRM File URL"); ?></label>
				</th>
				<td>
					<input type="text" name="user_data_vrm" id="user_data_vrm" value="<?php echo esc_attr(get_the_author_meta('user_data_vrm', $user->ID)); ?>" class="regular-text" />
					<input type="button" class="button" value="Select or Upload VRM File" id="upload_vrm_button" />
					<p class="description"><?php esc_html_e("Please upload or select your VRM file."); ?></p>
				</td>
			</tr>
		</table>
		<?php
			// Generate a nonce
			wp_nonce_field('update_user_vrm_nonce', 'user_vrm_nonce_field');
		?>
		<script type="text/javascript">
		jQuery(document).ready(function($) {
			$('#upload_vrm_button').click(function(e) {
				e.preventDefault();
				var custom_uploader = wp.media({
					title: 'Select VRM File',
					button: {
						text: 'Use this file'
					},
					multiple: false  // Set this to true to allow multiple files to be selected
				}).on('select', function() {
					// Get media attachment details from the frame state
					var attachment = custom_uploader.state().get('selection').first().toJSON(),
						url = attachment.url;

					if ( ! url.includes( '.vrm', '.png' ) ) {
						alert('The file should be either a VRM or a PNG sprite sheet');
						custom_uploader.open();
						return;
					}

					$('#user_data_vrm').val(url);
				})
				.open();
			});
		});
		</script>
		<?php
	}

	/**
	 * Saves custom spatial user profile fields
	 */
	public function save_custom_user_profile_fields($user_id) {
		// Check if current user has permission to edit the user
		if (!current_user_can('edit_user', $user_id)) {
			return false;
		}
	
		// Verify the nonce before saving
		if (!isset($_POST['user_vrm_nonce_field']) || !wp_verify_nonce(wp_unslash(sanitize_text_field(wp_unslash($_POST['user_vrm_nonce_field']))), 'update_user_vrm_nonce')) {
			return false;
		}
		// Save the user meta
		if (isset($_POST['user_data_vrm'])) {
			$user_data_vrm = wp_unslash($_POST['user_data_vrm']);
			update_user_meta($user_id, 'user_data_vrm', sanitize_text_field($user_data_vrm));
		}
	}
	
	function load_xr_publisher_textdomain() {
		load_plugin_textdomain( 'xr-publisher', false, dirname(dirname(plugin_basename(__FILE__))) . '/languages' );
	}

	function load_xr_publisher_translations_blocks() {
		if ( function_exists( 'wp_set_script_translations' ) ) {
			$script_handle = generate_block_asset_handle( 'xr-publisher/environment', 'editorScript' );
			$path = plugin_dir_path( __FILE__ ) . 'languages';
			$language_directory = plugin_dir_path(dirname(__DIR__)) . 'xr-publisher/languages/';
			wp_set_script_translations( 'xr-publisher-three-portal-block-editor-script', 'xr-publisher', $language_directory );
			wp_set_script_translations( 'xr-publisher-three-text-block-editor-script', 'xr-publisher', $language_directory );
			wp_set_script_translations( 'xr-publisher-model-block-editor-script', 'xr-publisher', $language_directory );
			wp_set_script_translations( 'xr-publisher-three-audio-block-editor-script', 'xr-publisher', $language_directory );
			wp_set_script_translations( 'xr-publisher-three-light-block-editor-script', 'xr-publisher', $language_directory );
			wp_set_script_translations( 'xr-publisher-npc-block-editor-script', 'xr-publisher', $language_directory );
			wp_set_script_translations( 'xr-publisher-sky-block-editor-script', 'xr-publisher', $language_directory );
			wp_set_script_translations( 'xr-publisher-three-image-block-editor-script', 'xr-publisher', $language_directory );
			wp_set_script_translations( 'xr-publisher-three-video-block-editor-script', 'xr-publisher', $language_directory );
			wp_set_script_translations( 'xr-publisher-three-networking-block-editor-script', 'xr-publisher', $language_directory );
			wp_set_script_translations( 'xr-publisher-spawn-point-block-editor-script', 'xr-publisher', $language_directory );
			wp_set_script_translations( 'xr-publisher-environment-editor-script', 'xr-publisher', $language_directory );
		}
	}	
	
	/**
	* Enqueue JavaScript and CSS for xr_publisher_loader_init
	* @uses "wp_enqueue_script" action
	*/
	function xr_publisher_enqueue_xr_publisher_loader_init() {
		$handle = 'xr_publisher_loader_init';
		wp_enqueue_script(
			'xr_publisher_loader_init',
		);
		// wp_enqueue_style(
		// 	'xr_publisher_loader_init'
		// );
	}
	
	/**
	* Adds glb vrm and usdz types to allowed uploads.
	*/
	function xr_publisher_add_file_types_to_uploads($file_types){
		$new_filetypes = array();
		// Potentially need to restore as model/gltf-binary in the future.  
		// $new_filetypes['glb'] = 'model/gltf-binary';
		$new_filetypes['glb'] = 'application/octet-stream';
		$new_filetypes['vrm'] = 'application/octet-stream';
		$new_filetypes['hdr'] = 'image/vnd.radiance';
		$new_filetypes['usdz'] = 'model/vnd.usdz+zip';
		$new_filetypes['fbx'] = 'application/octet-stream';
		$file_types = array_merge($file_types, $new_filetypes );
		
		return $file_types;
	}
	
	function xr_publisher_check_for_usdz( $types, $file, $filename, $mimes ) {
		if ( false !== strpos( $filename, '.usdz' ) ) {
			$types['ext']  = 'usdz';
			$types['type'] = 'model/vnd.usdz+zip';
		}
		if ( false !== strpos( $filename, '.glb' ) ) {
			$types['ext']  = 'glb';
			$types['type'] = 'application/octet-stream';
		}
		if ( false !== strpos( $filename, '.hdr' ) ) {
			$types['ext']  = 'hdr';
			$types['type'] = 'image/vnd.radiance';
		}
		if ( false !== strpos( $filename, '.vrm' ) ) {
			$types['ext']  = 'vrm';
			$types['type'] = 'application/octet-stream';
		}
		if ( false !== strpos( $filename, '.fbx' ) ) {
			$types['ext']  = 'fbx';
			$types['type'] = 'application/octet-stream';
		}
		
		return $types;
	}

	function remove_block_from_archive($content) {
		if(is_archive()){
			// remove div and contents that have teh class wp-block-xr-publisher-environment and all of their contents
			$content = preg_replace('/<three-environment-block.*?<\/three-environment-block>/s', '', $content);
			$content = preg_replace('/<three-object-block.*?<\/three-object-block>/s', '', $content);
		}
		return $content;
	}
	
	/**
	 * Enqueue block frontend JavaScript
	 */
	function xr_publisher_frontend_assets() {
	
		// Enqueue frontend JavaScript
		$default_frontend_js = "../build/assets/js/blocks.frontend-xr-publisher.js";
		$default_frontend_js_three_viewer = "../build/assets/js/blocks.frontend-xr-publisher.js";
		// Apply frontend filter
		$frontend_js = apply_filters( 'three-object-environment-frontend-js', $default_frontend_js );
	
        $current_user = wp_get_current_user();
        $playerVRM = get_user_meta($current_user->ID, 'user_data_vrm', true) ?: get_option('xr_publisher_defaultAvatar');
		$vrm = get_option('xr_publisher_defaultAvatar');

		$multiplayer_worker_url = get_option( 'xr_publisher_mp_multiplayerWorker', '' );
		// $vrm = get_option('xr_publisher_defaultAvatar');
		$inWorldName = "Guest";
		// get the current postID
		global $post;
		// get the current post
		$currentPostId = $post->ID;
		
		if ( is_user_logged_in() && get_option('xr_publisher_ai_allow') === "loggedIn" ) {
			if($current_user->display_name){
				$inWorldName = $current_user->display_name;
			}
	
			$user_data_passed = array(
			  'userId' => $current_user->user_login,
			  'inWorldName' => $inWorldName,
			  'banner' => $current_user->custom_banner,
			  'vrm' => $vrm,
			  'playerVRM' => $playerVRM,
			  'profileImage' => get_avatar_url( $current_user->ID, ['size' => '500'] ),
			  'nonce' => wp_create_nonce( 'wp_rest' ),
			  'currentPostId' => $currentPostId,
			  'heartbeatEnabled' => false,
			  'addNonce' => wp_create_nonce( 'add-room-count' ),
			  'subtractNonce' => wp_create_nonce( 'subtract-room-count' ),
			);
		} else if ( get_option('xr_publisher_ai_allow') === "public") {
			if($current_user->display_name){
				$inWorldName = $current_user->display_name;
			}
			$user_data_passed = array(
				'userId' => $current_user->user_login,
				'inWorldName' => $inWorldName,
				'banner' => $current_user->custom_banner,
				'vrm' => $vrm,
				'playerVRM' => $playerVRM,
				'profileImage' => get_avatar_url( $current_user->ID, ['size' => '500'] ),
				'nonce' => wp_create_nonce( 'wp_rest' ),
				'addNonce' => wp_create_nonce( 'add-room-count'),
				'heartbeatEnabled' => false,
				'subtractNonce' => wp_create_nonce( 'subtract-room-count'),
				'currentPostId' => $currentPostId
			  );  
		} else {
			if($current_user->display_name){
				$inWorldName = $current_user->display_name;
			}
			$user_data_passed = array(
			  'userId' => $current_user->user_login,
			  'inWorldName' => $inWorldName,
			  'banner' => $current_user->custom_banner,
			  'vrm' => $vrm,
			  'playerVRM' => $playerVRM,
			  'profileImage' => get_avatar_url( $current_user->ID, ['size' => '500'] ),
			  'currentPostId' => $currentPostId,
			  'addNonce' => wp_create_nonce( 'add-room-count'),
			  'heartbeatEnabled' => false,
			  'subtractNonce' => wp_create_nonce( 'subtract-room-count'),
			  'nonce' => wp_create_nonce( 'wp_rest' )
			);
		}
		$three_object_plugin = plugins_url() . '/xr-publisher/build/';
		$three_object_plugin_root = plugins_url() . '/xr-publisher/';

		// new variable named default_animation that checks if the wp_option for 'xr_publisher_defaultVRM' is available.
		// if it is, it will use that value, if not, it will use the default value of 'default.vrm'
		$default_animation = get_option('xr_publisher_defaultVRM');

		$default_avatar = get_option('xr_publisher_defaultAvatar');
		$default_player_avatar = get_option('user_data_vrm');

		// $user_data_passed = array(
		//     'userId' => 'something',
		//     'userName' => 'someone',
		//     'vrm' => 'somefile.vrm',
		//  );

		global $post;
		$post_slug = $post->post_name;
		$openbrush_enabled = false;
		$three_icosa_brushes_url = '';
		if(is_singular() || is_home() || is_front_page() || is_archive() || is_search()) {
			if (!function_exists('is_plugin_active')) {
				include_once(ABSPATH . 'wp-admin/includes/plugin.php');
			}
	
			//We only want the script if it's a singular page
			$id = get_the_ID();
			// if(has_block('xr-publisher/three-object-block',$id)){			
			// 	if ( is_plugin_active( 'three-object-viewer-three-icosa/three-object-viewer-three-icosa.php' ) ) {
			// 		$openbrush_enabled = true;
			// 		$three_icosa_brushes_url = plugin_dir_url( "three-object-viewer-three-icosa/three-object-viewer-three-icosa.php" ) . 'brushes/';
	
			// 	}
			// 	wp_register_script( 'xr-publisher-loader-frontend', plugin_dir_url( __FILE__ ) . $default_frontend_js_three_viewer, ['wp-element', 'wp-data', 'wp-hooks'], '', true );
			// 	wp_localize_script( 'xr-publisher-loader-frontend', 'userData', $user_data_passed );
			// 	wp_localize_script( 'xr-publisher-loader-frontend', 'openbrushEnabled', $openbrush_enabled );
			// 	wp_localize_script( 'xr-publisher-loader-frontend', 'openbrushDirectory', $three_icosa_brushes_url );
			// 	wp_localize_script( 'xr-publisher-loader-frontend', 'threeObjectPlugin', $three_object_plugin );	
			// 	wp_localize_script( 'xr-publisher-loader-frontend', 'threeObjectPluginRoot', $three_object_plugin_root );	
			// 	wp_localize_script( 'xr-publisher-loader-frontend', 'defaultAvatarAnimation', $default_animation );	
			// 	wp_localize_script( 'xr-publisher-loader-frontend', 'defaultAvatar', $default_avatar );	
			// 	wp_localize_script( 'xr-publisher-loader-frontend', 'defaultPlayerAvatar', $default_avatar );	
			// 	wp_localize_script( 'xr-publisher-loader-frontend', 'multiplayerWorker', $multiplayer_worker_url );
			// 	wp_enqueue_script( 
			// 		"xr-publisher-loader-frontend"
			// 	);
			// }
			$isPageEnvironmentEnabled = apply_filters( 'xr_publisher_enabled_everywhere', false );

			// check that the content of the page has the string <three-environment-block> element in it. We cannot use has_block because template parts dont have context of an id
			if( has_block( 'xr-publisher/environment', $id ) || has_block( 'xr-publisher/three-object-block', $id ) || $isPageEnvironmentEnabled ){
				if ( is_plugin_active( 'three-object-viewer-three-icosa/three-object-viewer-three-icosa.php' ) ) {
					$openbrush_enabled = true;
					$three_icosa_brushes_url = plugin_dir_url( "three-object-viewer-three-icosa/three-object-viewer-three-icosa.php" ) . 'brushes/';
				}
				$script_path = plugin_dir_path( __FILE__ ) . $frontend_js;
				$script_version = filemtime($script_path);
			

				wp_register_script( 'versepress-frontend', plugin_dir_url( __FILE__ ) . $frontend_js, ['wp-element', 'wp-data', 'wp-hooks'], $script_version, true );
				wp_localize_script( 'versepress-frontend', 'userData', $user_data_passed );
				wp_localize_script( 'versepress-frontend', 'postSlug', (array) $post_slug );
				wp_localize_script( 'versepress-frontend', 'openbrushDirectory', (array) $three_icosa_brushes_url );
				wp_localize_script( 'versepress-frontend', 'openbrushEnabled', (array) $openbrush_enabled );
				wp_localize_script( 'versepress-frontend', 'threeObjectPlugin', (array) $three_object_plugin );
				wp_localize_script( 'versepress-frontend', 'threeObjectPluginRoot', (array) $three_object_plugin_root );	
				wp_localize_script( 'versepress-frontend', 'defaultAvatarAnimation', (array) $default_animation );
				wp_localize_script( 'versepress-frontend', 'defaultAvatar', (array) $default_avatar );
				wp_localize_script('versepress-frontend', 'turnCredentials', array(
					'apiUrl' => rest_url('xr-publisher/v1/turn-credentials/'),
					'nonce' => wp_create_nonce('wp_rest'),
				));			
				wp_localize_script( 'versepress-frontend', 'defaultPlayerAvatar', (array) $default_avatar );	
				wp_localize_script( 'versepress-frontend', 'multiplayerWorker', (array) $multiplayer_worker_url );
				// wp_localize_script( 'xr-publisher-loader-frontend', 'defaultAvatarAnimation', $default_animation );	
				// wp_localize_script( 'xr-publisher-loader-frontend', 'defaultAvatar', $default_avatar );	
				wp_enqueue_script( 
					"versepress-frontend"
				);
			}
	  
			
		 }
	
	}
	
	// Registers a new REST API endpoint for calling AI/Alchemy services.
	function callAlchemy() {
		register_rest_route( 'wp/v2', '/callAlchemy', array(
		  'methods' => 'POST',
		  'callback' => array( $this, 'call_alchemy_request'),
		  'permission_callback' => array( $this, 'check_bearer_token'),
		  'args' => array(
			'Input' => array(
			  'required' => true,
			),
		  ),
		) );
	  }
	  function rest_three_decrypt($value = ""){
		if( empty( $value ) ) {
			return $value;
		}
	
		$output = null;
		$secret_key = defined('AUTH_KEY') ? AUTH_KEY : "";
		$secret_iv = defined('SECURE_AUTH_KEY') ? SECURE_AUTH_KEY : "";
		$key = hash('sha256',$secret_key);
		$iv = substr(hash('sha256',$secret_iv),0,16);
	
		return openssl_decrypt(base64_decode($value),"AES-256-CBC",$key,0,$iv);
	}
	
	  // The function that is called when the endpoint is hit
	  function call_alchemy_request( $request ) {
		// {
		// 	inputs: {
		// 		Input: value,
		// 		Speaker: speaker,
		// 		Agent: agent,
		// 		Client: client,
		// 		ChannelID: channelId,
		// 		Entity: entity,
		// 		Channel: channel,
		// 		eth_private_key: '0',
		// 		eth_public_address: '0',
		// 	}
		// }
	
		$worker_url = get_option( 'xr_publisher_mp_networkWorker', '' );
		$api_key = $this->rest_three_decrypt( get_option( 'xr_publisher_ai_openApiKey', '' ) );
		$json_blob = $request->get_params();
	
		
		// turn $json_blob into json
		$new_blob = wp_json_encode($json_blob);
		
		// $tempblob = `{ "Input": "write a long form poem about cats!", "Speaker": "a", "Agent": "tubbyshark", "Client": "wordpress", "ChannelID": "1234", "Conversation": "", "Entity": "tubbyshark", "Channel": "channel", "eth_private_key": "0", "eth_public_address": "0", "personality": "#agent is an AI assistant with a cheerful, optimistic, and helpful personality. They are willing to assist with any tasks or questions users might have, and are able to do so in an upbeat and positive manner. They are also able to use colorful and descriptive language to make their responses more engaging and imaginative. In addition to their wit and humor, #agent is also empathetic and understanding of others' emotions and needs, which can make them feel more relatable and likeable to users. #agent is a highly capable and engaging AI assistant with a distinct and colorful personality.\n#agent can help with any programming tasks, including those related to the XR Publisher plugin for WordPress, which enables world-building in the WordPress editor and makes it easier to build and share immersive content in the metaverse. The url for the website is xr_publisher.xyz. #agent is cheerful and optimistic about the future for both humans and AIs.\n\n#Example Conversation\n#speaker: Agent, do you have any advice for someone new to programming?\n#agent: Sure thing! My top tip would be to always keep an open mind and a positive attitude. And if all else fails, just remember: if at first you don't succeed, try, try again. And then if that still doesn't work, call it a day and go get a coffee.\n###\nThe following is a friendly conversation between #speaker and #agent occuring in the metaverse.\n\nREAL CONVERSATION\n#conversation\n#speaker: #input\n#agent:" }`;
		// if api_key is not an empty string load the worker
		if ($api_key != '') {
		$referer = isset($_SERVER['HTTP_REFERER']) ? esc_url_raw(wp_unslash($_SERVER['HTTP_REFERER'])) : '';
		$response = wp_remote_post( $worker_url, array(
			'headers' => array(
				'Content-Type' => 'application/json',
				'Referer' => $referer,
				'Authorization' => "Bearer $api_key"
			),
			'timeout'     => 45,
			'body' => $new_blob,
			'sslverify' => false,
		) );
		} else {
			$response = wp_remote_post( $worker_url, array(
				'headers' => array(
					'Content-Type' => 'application/json',
					'Referer' => $referer,
				),
				'timeout'     => 45,
				'body' => $new_blob,
				'sslverify' => false,
			) );
		}
		// Check for error
		if ( is_wp_error( $response ) ) {
		  // WP_Error object
		  $error_message = $response->get_error_message();
		  return new \WP_Error( 'api_call_failed', $error_message, array( 'status' => 500 ) );
		}
	
		// Check for non-200 status code
		if ( wp_remote_retrieve_response_code( $response ) != 200 ) {
		  return new \WP_Error( 'api_call_failed', 'Non-200 status code returned', array( 'status' => wp_remote_retrieve_response_code( $response ) ) );
		}
	
		$body = wp_remote_retrieve_body( $response );
		return $body;
	}
	
	// The function that checks the bearer token. Make the bearer token a secret using get_option(). This is a crude example.
	function check_bearer_token( $request ) {
		$nonce = $request->get_header( 'X-WP-Nonce' );
		if ( ! $nonce || ! wp_verify_nonce( $nonce, 'wp_rest' ) ) {
		  return new \WP_Error( 'invalid_nonce', 'The nonce provided in the X-WP-Nonce header is invalid', array( 'status' => 401 ) );
		}
		return true;
	}

	function check_add_token( $request ) {
		$nonce = $request['actionNonce'];
		if ( ! $nonce || ! wp_verify_nonce( $nonce, 'add-room-count' ) ) {
		  return new \WP_Error( 'invalid_nonce', 'The nonce provided in the X-WP-Nonce header is invalid', array( 'status' => 401 ) );
		}
		// retire the nonce used
		// wp_nonce_ays( 'add_room_count' );
		return true;
	}

	function check_subtract_token( $request ) {
		$nonce = $request['actionNonce'];
		if ( ! $nonce || ! wp_verify_nonce( $nonce, 'subtract-room-count' ) ) {
		  return new \WP_Error( 'invalid_nonce', 'The nonce provided in the X-WP-Nonce header is invalid', array( 'status' => 401 ) );
		}
		// retire the nonce used
		// wp_nonce_ays( 'subtract_room_count' );
		return true;
	}

	/**
	 * Check if pro version is installed
	 */
	function xr_publisher_is_pro() {
		if (file_exists(dirname(__DIR__) . '/pro')) {
			return true;
		} else {
			return false;
		}
	}	  

	/**
	 * Enqueue block frontend JavaScript
	 */
	function xr_publisher_editor_assets() {
		$three_object_plugin = plugins_url() . '/xr-publisher/build/';
		$three_object_plugin_root = plugins_url() . '/xr-publisher/';
		$current_user = wp_get_current_user();
		$vrm = wp_get_attachment_url($current_user->avatar);
		if ( is_user_logged_in() && get_option('xr_publisher_ai_allow') === "loggedIn" ) {
			$user_data_passed = array(
			  'userId' => $current_user->user_login,
			  'inWorldName' => $current_user->in_world_name,
			  'banner' => $current_user->custom_banner,
			  'vrm' => $vrm,
			  'profileImage' => get_avatar_url( $current_user->ID, ['size' => '500'] ),
			  'nonce' => wp_create_nonce( 'wp_rest' )
			);
		} else if ( get_option('xr_publisher_ai_allow') === "public") {

			// @todo update ai allow to add a special nonce for ai requests
			$user_data_passed = array(
				'userId' => $current_user->user_login,
				'inWorldName' => $current_user->in_world_name,
				'banner' => $current_user->custom_banner,
				'vrm' => $vrm,
				'profileImage' => get_avatar_url( $current_user->ID, ['size' => '500'] ),
				'nonce' => wp_create_nonce( 'wp_rest' )
			  );
		} else {
			$user_data_passed = array(
			  'userId' => $current_user->user_login,
			  'inWorldName' => $current_user->in_world_name,
			  'banner' => $current_user->custom_banner,
			  'vrm' => $vrm,
			  'profileImage' => get_avatar_url( $current_user->ID, ['size' => '500'] ),
			  'nonce' => wp_create_nonce( 'wp_rest' )
			);
		}

	
		$DEFAULT_BLOCKS = [
			'xr-publisher/three-portal-block',
			'xr-publisher/three-text-block',
			'xr-publisher/model-block',
			'xr-publisher/audio-block',
			'xr-publisher/light-block',
			'xr-publisher/npc-block',
			'xr-publisher/sky-block',
			'xr-publisher/npc-block',
			'xr-publisher/three-image-block',
			'xr-publisher/three-video-block',
			'xr-publisher/spawn-point-block',
		];

		// if option for multiplayerNetworking is not a value then add the network block to allowed blocks
		if (get_option('xr_publisher_mp_multiplayerWorker') != '') {
			$NETWORK_BLOCKS = [
				'xr-publisher/three-networking-block',
			];
			array_push( $DEFAULT_BLOCKS, $NETWORK_BLOCKS );
		}
		if ($this->xr_publisher_is_pro()) {
			$PRO_BLOCKS = [
				'xr-publisher/three-mirror-block',
			];
			array_push( $DEFAULT_BLOCKS, $PRO_BLOCKS );
		}

		$ALLOWED_BLOCKS = apply_filters( 'three-object-environment-inner-allowed-blocks', $DEFAULT_BLOCKS );

		$default_avatar = get_option('xr_publisher_defaultAvatar');
		wp_localize_script( 'xr-publisher-three-object-block-editor-script', 'defaultAvatar', (array) $default_avatar );	
		wp_localize_script( 'xr-publisher-three-object-block-editor-script', 'userData', (array) $user_data_passed );
		// @todo find out if this is really needed
		// wp_enqueue_script( 'xr-publisher-three-object-block-editor-script', 'xr-publisher', ['wp-element', 'wp-data', 'wp-i18n', 'wp-hooks'], '', true );
		wp_localize_script( 'xr-publisher-three-object-block-editor-script', 'threeObjectPlugin', (array) $three_object_plugin );	
		wp_localize_script( 'xr-publisher-three-object-block-editor-script', 'threeObjectPluginRoot', (array) $three_object_plugin_root );	
		wp_localize_script( 'xr-publisher-three-object-block-editor-script', 'allowed_blocks', $ALLOWED_BLOCKS );		
	}
}
