import { useState, useEffect } from "@wordpress/element";
import { __, sprintf } from "@wordpress/i18n";
import { Button, Spinner, ExternalLink } from "@wordpress/components";

export default function App({ getSettings, updateSettings }) {
  const [settings, setSettings] = useState({});
  const [isLoading, setIsLoading] = useState(true);
  const [isSaving, setIsSaving] = useState(false);
  const [isOpenApiKeyVisible, setIsOpenApiKeyVisible] = useState(false);
  const [mediaFrame, setMediaFrame] = useState(null);

  useEffect(() => {
    getSettings().then((response) => {
      setSettings(response);
      setIsLoading(false);
    });
  }, [getSettings]);

  const onSave = async (event) => {
    event.preventDefault();
    setIsSaving(true);
    await updateSettings(settings);
    setIsSaving(false);
  };

  const openMediaUploader = (settingKey) => {
    if (mediaFrame) {
      mediaFrame.open();
      return;
    }

    const frame = wp.media({
      title: __("Select or Upload Media", "xr-publisher"),
      button: {
        text: __("Use this media", "xr-publisher"),
      },
      multiple: false,
    });

    frame.on("select", () => {
      const attachment = frame.state().get("selection").first().toJSON();
      setSettings({ ...settings, [settingKey]: attachment.url });
    });

    setMediaFrame(frame);
    frame.open();
  };

  const clearMedia = (settingKey) => {
    setSettings({ ...settings, [settingKey]: "" });
  };

  if (isLoading) {
    return <Spinner />;
  }

  return (
    <div className="wrap">
      <h1>{__("XR Publisher Settings", "xr-publisher")}</h1>
      <p>
        {__(
          "Here you can manage the settings for XR Publisher to tweak global configuration options and save your API keys for connected services.",
          "xr-publisher"
        )}
      </p>

      <form onSubmit={onSave}>
        <table className="form-table" role="presentation">
          <tbody>
            <tr>
              <th scope="row">
                <label htmlFor="xrPublisherApiKey">{__("XR Publisher Pro API Key", "xr-publisher")}</label>
              </th>
              <td>
                <input
                  type="text"
                  id="xrPublisherApiKey"
                  value={settings.xrPublisherApiKey || ""}
                  onChange={(event) => setSettings({ ...settings, xrPublisherApiKey: event.target.value })}
                  className="regular-text"
                />
                <p className="description">
                  {
                    <ExternalLink href="https://app.xr.foundation">
                      {__("Sign up for Pro", "xr-publisher")}
                    </ExternalLink>
                  }
                </p>
              </td>
            </tr>

            <tr>
              <th scope="row">
                <label htmlFor="toyboxApiKey">{__("Toybox API Key", "xr-publisher")}</label>
              </th>
              <td>
                <input
                  type="text"
                  id="toyboxApiKey"
                  value={settings.toyboxApiKey || ""}
                  onChange={(event) => setSettings({ ...settings, toyboxApiKey: event.target.value })}
                  className="regular-text"
                />
                <p className="description">
                  {
					<>
						<ExternalLink href="https://app.xr.foundation">
						{__("Sign up for Toybox Storage", "xr-publisher")}
						</ExternalLink>
					</>
                  }
                </p>
              </td>
            </tr>

            <tr>
              <th scope="row">{__("Default Animation", "xr-publisher")}</th>
              <td>
                <input
                  type="text"
                  value={settings.defaultVRM || ""}
                  readOnly
                  className="regular-text"
                />
                <Button
                  isPrimary
                  onClick={() => openMediaUploader("defaultVRM")}
                >
                  {__("Set Default Animation", "xr-publisher")}
                </Button>
                {settings.defaultVRM && (
                  <Button
                    isSecondary
                    onClick={() => clearMedia("defaultVRM")}
                  >
                    {__("Clear Default Animation", "xr-publisher")}
                  </Button>
                )}
              </td>
            </tr>

            <tr>
              <th scope="row">{__("Default Avatar", "xr-publisher")}</th>
              <td>
                <input
                  type="text"
                  value={settings.defaultAvatar || ""}
                  readOnly
                  className="regular-text"
                />
                <p className="description">
                  {<ExternalLink href="https://3ov.xyz/resource/recommended-avatars/">
                      {__("Compatible Avatars", "xr-publisher")}
                    </ExternalLink>}
                </p>
                <Button
                  isPrimary
                  onClick={() => openMediaUploader("defaultAvatar")}
                >
                  {__("Set Default Avatar", "xr-publisher")}
                </Button>
                {settings.defaultAvatar && (
                  <Button
                    isSecondary
                    onClick={() => clearMedia("defaultAvatar")}
                  >
                    {__("Clear Default Avatar", "xr-publisher")}
                  </Button>
                )}
              </td>
            </tr>
			{settings.xrPublisherApiKey && (
				<>
					<tr>
						<th scope="row">
							<label htmlFor="multiplayerWorker">{__("Networking Endpoint URL", "xr-publisher")}</label>
						</th>
						<td>
							<input
							type="text"
							id="multiplayerWorker"
							value={settings.multiplayerWorker || ""}
							onChange={(event) => setSettings({ ...settings, multiplayerWorker: event.target.value })}
							className="regular-text"
							/>
							<p className="description">
							{__(
								"Use https://p2pcf.sxp.digital/ or host your own CloudFlare Worker using ",
								"xr-publisher"
							)}
							<ExternalLink href="https://github.com/gfodor/p2pcf">
								p2pcf
							</ExternalLink>
							{__(
								". A tutorial for setting up your own worker can be found ",
								"xr-publisher"
							)}
							<ExternalLink href="https://github.com/gfodor/p2pcf/blob/master/INSTALL.md">
								{__("here", "xr-publisher")}
							</ExternalLink>.
							</p>
						</td>
						</tr>
					<tr>
					<th scope="row">{__("TURN Settings", "xr-publisher")}</th>
					<td>
						<p className="description">
						{__(
							"These settings are used to configure the TURN server for WebRTC connections. You can use the public TURN server or host your own. The public TURN server is hosted at ",
							"xr-publisher"
						)}
						<code>turn.sxp.digital</code>
						{__(" but is limited.", "xr-publisher")}
						</p>
						<p className="description">
						{__(
							"A TURN server is used to relay WebRTC connections when a direct connection cannot be established. This is common when two peers are behind different NATs or firewalls.",
							"xr-publisher"
						)}
						</p>
					</td>
					</tr>

					<tr>
					<th scope="row">
						<label htmlFor="turnCredentialRelay">{__("TURN Credential Relay", "xr-publisher")}</label>
					</th>
					<td>
						<input
						type="text"
						id="turnCredentialRelay"
						value={settings.turnCredentialRelay || ""}
						onChange={(event) => setSettings({ ...settings, turnCredentialRelay: event.target.value })}
						className="regular-text"
						/>
						<p className="description">
						{__(
							"A CloudFlare Worker is used to relay TURN credentials. You can use the public worker at ",
							"xr-publisher"
						)}
						<code>https://turn.sxp.digital/</code>
						{__(
							" or host your own. The public worker is resource limited and should only be used for testing. You can bypass these limits using your API key below or by hosting your own CloudFlare Worker to run the exact same credential handling. A tutorial for setting up your own worker can be found ",
							"xr-publisher"
						)}
						<ExternalLink href="https://github.com/gfodor/p2pcf/blob/master/INSTALL.md">
							{__("here", "xr-publisher")}
						</ExternalLink>.
						</p>
					</td>
					</tr>

					<tr>
					<th scope="row">
						<label htmlFor="turnServerKey">{__("TURN API Key", "xr-publisher")}</label>
					</th>
					<td>
						<input
						type={isOpenApiKeyVisible ? "text" : "password"}
						id="turnServerKey"
						value={settings.turnServerKey || ""}
						onChange={(event) => setSettings({ ...settings, turnServerKey: event.target.value })}
						className="regular-text"
						/>
						<Button
						isSecondary
						onClick={() => setIsOpenApiKeyVisible(!isOpenApiKeyVisible)}
						>
						{isOpenApiKeyVisible
							? __("Hide Key", "xr-publisher")
							: __("Show Key", "xr-publisher")
						}
						</Button>
						<p className="description">
						{__(
							"This secret key will bypass limitations of the XR Publisher public TURN worker. You can use ",
							"xr-publisher"
						)}
						<ExternalLink href="https://www.metered.ca/">
							metered.ca
						</ExternalLink>
						{__(
							" to establish a secret key. More info can be found ",
							"xr-publisher"
						)}
						<ExternalLink href="https://www.metered.ca/docs/turnserver-guides/expiring-turn-credentials/#step-1-obtain-your-metered-domain-and-secret-key">
							{__("here", "xr-publisher")}
						</ExternalLink>.
						</p>
					</td>
					</tr>

					<tr>
					<th scope="row">
						<label htmlFor="multiplayerAccess">{__("Multiplayer Access", "xr-publisher")}</label>
					</th>
					<td>
						<select
						id="multiplayerAccess"
						value={settings.multiplayerAccess || "loggedIn"}
						onChange={(event) => setSettings({ ...settings, multiplayerAccess: event.target.value })}
						>
						<option value="loggedIn">{__("TURN for Logged In Only", "xr-publisher")}</option>
						<option value="public">{__("TURN Allowed for Public", "xr-publisher")}</option>
						</select>
					</td>
					</tr>
				</>
			)}
            <tr>
              <th scope="row">{__("NPC Settings", "xr-publisher")}</th>
              <td>
                <label htmlFor="enabled">
                  <input
                    type="checkbox"
                    id="enabled"
                    checked={settings.enabled || false}
                    onChange={(event) => setSettings({ ...settings, enabled: event.target.checked })}
                  />
                  {__("Enable NPC", "xr-publisher")}
                </label>
              </td>
            </tr>

            <tr>
              <th scope="row">
                <label htmlFor="networkWorker">{__("AI Endpoint URL", "xr-publisher")}</label>
              </th>
              <td>
                <input
                  type="text"
                  id="networkWorker"
                  value={settings.networkWorker || ""}
                  onChange={(event) => setSettings({ ...settings, networkWorker: event.target.value })}
                  className="regular-text"
                />
              </td>
            </tr>

            <tr>
              <th scope="row">
                <label htmlFor="openApiKey">{__("OpenAI API Token", "xr-publisher")}</label>
              </th>
              <td>
                <input
                  type={isOpenApiKeyVisible ? "text" : "password"}
                  id="openApiKey"
                  value={settings.openApiKey || ""}
                  onChange={(event) => setSettings({ ...settings, openApiKey: event.target.value })}
                  className="regular-text"
                />
                <Button
                  isSecondary
                  onClick={() => setIsOpenApiKeyVisible(!isOpenApiKeyVisible)}
                >
                  {isOpenApiKeyVisible
                    ? __("Hide Key", "xr-publisher")
                    : __("Show Key", "xr-publisher")
                  }
                </Button>
              </td>
            </tr>

            <tr>
              <th scope="row">
                <label htmlFor="allowPublicAI">{__("AI Access Level", "xr-publisher")}</label>
              </th>
              <td>
                <select
                  id="allowPublicAI"
                  value={settings.allowPublicAI || "loggedIn"}
                  onChange={(event) => setSettings({ ...settings, allowPublicAI: event.target.value })}
                >
                  <option value="public">{__("Public", "xr-publisher")}</option>
                  <option value="loggedIn">{__("Logged In", "xr-publisher")}</option>
                </select>
              </td>
            </tr>
          </tbody>
        </table>

        <p className="submit">
          <Button
            isPrimary
            type="submit"
            isBusy={isSaving}
          >
            {isSaving
              ? __("Saving...", "xr-publisher")
              : __("Save Settings", "xr-publisher")
            }
          </Button>
        </p>
      </form>
    </div>
  );
}